package com.ync365.jrpt.service.pagination;

import com.ync365.jrpt.util.JacksonMapperUtil;
import com.ync365.jrpt.util.StringUtil;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 分页参数封装类.
 */
public class PageRequestDTO implements java.io.Serializable
{


	private static final long serialVersionUID = 3276329216554067912L;

	/**
     * 第几页
     */
    protected int pageNo = 1;

    /**
     * 每页大小
     */
    protected int pageSize = 10;

    /**
     * 排序字段
     */
    protected String orderBy = null;

    /**
     * 排序方式
     *
     * @see Sort
     */
    protected String orderDir = null;

    /**
     * 是否默认计算总记录数.
     */
    protected boolean countTotal = true;

    // 分页列表大小，默认5页
    protected int sliderCount = 4;

    /**
     * 构造方法
     */
    public PageRequestDTO()
    {
    }

    /**
     * 构造方法
     *
     * @param pageNo   第几页
     * @param pageSize 每页分页大小
     */
    public PageRequestDTO(int pageNo, int pageSize)
    {
        this.pageNo = pageNo;
        this.pageSize = pageSize;
    }

    /**
     * 获得当前页的页号, 序号从1开始, 默认为1.
     *
     * @return
     */
    public int getPageNo()
    {
        return pageNo;
    }

    /**
     * 设置当前页的页号, 序号从1开始, 低于1时自动调整为1.
     *
     * @param pageNo
     */
    public void setPageNo(final int pageNo)
    {
        this.pageNo = pageNo;

        if (pageNo < 1)
        {
            this.pageNo = 1;
        }
    }

    /**
     * 获得每页的记录数量
     *
     * @return
     */
    public int getPageSize()
    {
        return pageSize;
    }

    /**
     * 设置每页的记录数量, 低于1时自动调整为1.
     *
     * @param pageSize
     */
    public void setPageSize(final int pageSize)
    {
        this.pageSize = pageSize;

        if (pageSize < 1)
        {
            this.pageSize = 1;
        }
    }

    /**
     * 获得排序字段, 无默认值. 多个排序字段时用','分隔.
     *
     * @return
     */
    public String getOrderBy()
    {
        return orderBy;
    }

    /**
     * 设置排序字段, 多个排序字段时用','分隔.
     *
     * @param orderBy
     */
    public void setOrderBy(final String orderBy)
    {
        this.orderBy = orderBy;
    }

    /**
     * 获得排序方向, 无默认值.
     *
     * @return
     */
    public String getOrderDir()
    {
        return orderDir;
    }

    /**
     * 设置排序方式向.
     *
     * @param orderDir 可选值为desc或asc,多个排序字段时用','分隔.
     */
    public void setOrderDir(final String orderDir)
    {
        if (StringUtil.isEmpty(orderDir))
        {
            return;
        }
        String lowcaseOrderDir = orderDir.toLowerCase();

        //检查order字符串的合法值
        String[] orderDirs = lowcaseOrderDir.split(",");

        for (String orderDirStr : orderDirs)
        {
            if (!Sort.DESC.equals(orderDirStr) && !Sort.ASC.equals(orderDirStr))
            {
                throw new IllegalArgumentException("Sort direction [ " + orderDirStr + " ] not legal value");
            }
        }

        this.orderDir = lowcaseOrderDir;
    }

    /**
     * 获得排序参数.
     *
     * @return
     */
    public List<Sort> getSort()
    {
        if (!isOrderBySetted())
        {
            return Collections.EMPTY_LIST;
        }

        String[] orderBys = orderBy.split(",");
        String[] orderDirs = orderDir.split(",");

        List<Sort> orders = new ArrayList<PageRequestDTO.Sort>();
        for (int i = 0; i < orderBys.length && orderBys.length == orderDirs.length; i++)
        {
            orders.add(new Sort(orderBys[i], orderDirs[i]));
        }

        return orders;
    }

    /**
     * 是否已设置排序字段,无默认值.
     *
     * @return
     */
    public boolean isOrderBySetted()
    {
        return (!StringUtil.isEmpty(orderBy) && !StringUtil.isEmpty(orderDir));
    }

    /**
     * 是否默认计算总记录数.
     *
     * @return
     */
    public boolean isCountTotal()
    {
        return countTotal;
    }

    /**
     * 设置是否默认计算总记录数.
     *
     * @param countTotal
     */
    public void setCountTotal(boolean countTotal)
    {
        this.countTotal = countTotal;
    }

    /**
     * 根据pageNo和pageSize计算当前页第一条记录在总结果集中的位置, 序号从0开始.
     *
     * @return
     */
    public int getOffset()
    {
        return ((pageNo - 1) * pageSize);
    }

    /**
     * 获取Order By后的sql字符串
     *
     * @return String
     */
    public String getOrderSortString()
    {
        List<Sort> list = this.getSort();
        StringBuilder buffer = new StringBuilder();
        for (Sort sort : list)
        {
            buffer.append(sort.property).append(" ").append(sort.dir).append(",");
        }
        if (buffer.length() > 0)
        {
            buffer.deleteCharAt(buffer.length() - 1);
        }
        return buffer.toString();
    }

    /**
     * 获取分页列表大小，默认5页
     *
     * @return
     */
    public int getSliderCount()
    {
        return sliderCount;
    }

    /**
     * 设置分页列表大小，默认5页
     *
     * @param sliderCount
     */
    public void setSliderCount(int sliderCount)
    {
        this.sliderCount = sliderCount;
    }

    @Override
    public String toString()
    {
        return JacksonMapperUtil.beanToJson(this);
    }

    public static class Sort
    {

        public static final String ASC = "asc";

        public static final String DESC = "desc";

        private final String property;

        private final String dir;

        /**
         * 构造方法
         *
         * @param property 要排序的字段
         * @param dir      排序方式
         */
        public Sort(String property, String dir)
        {
            this.property = property;
            this.dir = dir;
        }

        /**
         * 获取排序字段
         *
         * @return String
         */
        public String getProperty()
        {
            return property;
        }

        /**
         * 获取排序方式
         *
         * @return String
         */
        public String getDir()
        {
            return dir;
        }
    }
}
