package com.ync365.jrpt.business.biz;

import java.util.List;
import java.util.Map;
import java.util.Random;

import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.crypto.hash.SimpleHash;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ync365.jrpt.business.dao.SysUserDao;
import com.ync365.jrpt.business.dao.entity.SysUser;
import com.ync365.jrpt.business.pagination.Page;
import com.ync365.jrpt.business.pagination.PageBuilder;
import com.ync365.jrpt.business.pagination.PageRequest;
import com.ync365.jrpt.util.StringUtil;
/**
 * 
 * 功能描述: 系统用户管理
 * 
 * @version 1.0.0
 * @author 徐大伟
 */
@Service
public class SysUserBiz {
    @Autowired
    private SysUserDao sysUserDao;

    /**
     * 根据用户名查找用户
     * 
     * @param userName 用户名
     * @return
     */
    public SysUser findByUsername(String userName) {
        if (StringUtils.isBlank(userName)) {
            return null;
        }
        return sysUserDao.findByUserName(userName);
    }

    public int countSysUserByRoleId(int roleId) {
        return sysUserDao.countUserByRoleId(roleId);
    }

    /**
     * 
     * @param type 查询条件类型(user_name,real_name,mobile,userType)
     * @param value查询条件值
     * @param pageRequest
     * @return
     */
    public Page<Map<String, Object>> findUserList(String userName,
                                                  String realName,
                                                  String mobile,
                                                  String userType,
                                                  PageRequest pageRequest) {
        List<Map<String, Object>> list = sysUserDao.findUserList(userName, realName, mobile, userType, pageRequest.getOffset(), pageRequest.getPageSize());
        int total = sysUserDao.countUserList(userName, realName, mobile, userType, pageRequest.getOffset(), pageRequest.getPageSize());
        return PageBuilder.buildPage(pageRequest, list, total);
    }

    public int insert(SysUser user) {
        SysUser userTmp = sysUserDao.findByUserName(user.getRealName());
        if (userTmp != null) {
            return -1;
        } else {
            // 加密密码
            user.setPassword(encryptPwd(user.getPassword()));
            return sysUserDao.insert(user);
        }
    }

    /**
     * 逻辑删除用户
     * 
     * @param userName
     */
    public boolean removeUser(Integer id) {
        return (id!=null&&sysUserDao.removeUser(id)>0)?true:false;
    }

    public void updateUser(SysUser user) {
        sysUserDao.updateUser(user);
    }

    public SysUser selectByPrimaryKey(String id) {
        return sysUserDao.selectByPrimaryKey(id);
    }

    /**
     * 统计指定手机号的用户数量，包括已锁定
     * 
     * @param mobile 手机号
     * @return
     */
    public int countUserByMobile(String mobile) {
        return sysUserDao.countUserByMobile(mobile);
    }

    /**
     * 生成随机密码并更新指定用户
     * 
     * @return 密码明文
     */
    public String updateRandomPwd(int id) {
        String npwd = getStringRandom(6);
        sysUserDao.updatePasswordById(encryptPwd(npwd), id);
        return npwd;
    }

    public int updatePwd(int userId, String newPwd) {
        return sysUserDao.updatePasswordById(encryptPwd(newPwd), userId);
    }

    /**
     * 生成随机数字和字母,
     * 
     * @param length
     * @return
     */
    private String getStringRandom(int length) {
        String val = "";
        Random random = new Random();
        // 参数length，表示生成几位随机数
        for (int i = 0; i < length; i++) {
            String charOrNum = random.nextInt(2) % 2 == 0 ? "char" : "num";
            // 输出字母还是数字
            if ("char".equalsIgnoreCase(charOrNum)) {
                // 输出是大写字母还是小写字母
                int temp = random.nextInt(2) % 2 == 0 ? 65 : 97;
                val += (char) (random.nextInt(26) + temp);
            } else if ("num".equalsIgnoreCase(charOrNum)) {
                val += String.valueOf(random.nextInt(10));
            }
        }
        return val;
    }

    private String encryptPwd(String pwd) {
        return new SimpleHash("MD5", pwd).toHex();
    }

    /**
     * 保存前进行校验判断，如果返回5说明更新的手机号已存在，不更新
     * 
     * @param user
     * @return
     */
    public int checkAndSaveUser(SysUser user,
                                String oldPassword) {
        String userName = user.getRealName();
        SysUser oldUser = sysUserDao.findByUserName(userName);
        if (StringUtil.isNotBlank(oldPassword)) {
            String password = encryptPwd(user.getPassword());
            String pwd = encryptPwd(oldPassword);
            user.setPassword(password);
            if (!pwd.equals(oldUser.getPassword())) {
                return 6;
            }
        }
        String oldMobile = oldUser.getMobile();
        String newMobile = user.getMobile();
        int mobileCount = 0;
        if (!oldMobile.equals(newMobile)) {
            mobileCount = sysUserDao.countUserByMobile(newMobile);
            if (mobileCount > 0) {
                return 5;
            }
        }
        sysUserDao.updateUser(user);
        return 0;
    }

    public List<Map<String, Object>> findUserListByType(String type) {
        return sysUserDao.findUserListByType(type);
    }
}
