package com.jxwk.sso.client;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.jxwk.sso.client.util.SsoRemoteService;
import com.ync365.sso.service.dto.SessionDTO;
import com.ync365.sso.util.ConstantUtil;
import com.ync365.sso.util.ConstantUtil.AccessPolicy;
import com.ync365.sso.util.CookieUtil;
import com.ync365.sso.util.RequestUtil;
import com.ync365.sso.util.StringUtil;

import net.sf.ehcache.Element;

/**
 * 
 * SSO验证业务逻辑<br>
 * 
 * @author chentianyu
 */
public class AuthenticationService {

    /**
     * 
     * 功能描述: 初始化验证业务逻辑类<br>
     *
     * @return
     */
    public void init() {

    }

    /**
     * 
     * 功能描述: 销毁方法<br>
     *
     */
    public void destory() {

    }

    /**
     * 
     * 功能描述: <br>
     * 1.检测当前请求地址中是否包含yncsid 1.1. 包含：到redis中读取对应的sessionDTO，并将yncsid、yncuid、yncst写入到当前cookie中 2.检测当前域下是否包含yncsid 2.1.
     * 包含：到redis中读取对应的sessionDTO，判断sessionDTO是否为空 2.1.1. 不空：放行受限资源 2.1.2. 空：重定向到认证中心登录页面 3.检测当前请求是否是https请求 3.1.
     * 是：检测当前请求地址中是否包含yncst 3.2.1. 不包含：检测当前cookie中是否包含yncst 3.2.1.1. 包含：与sessionDTO中的yncst对比是否一致 3.2.1.1.1. 一致：放行受限资源
     * 3.2.1.1.2. 不一致：重定向到认证中心登录页面 3.2.1.2. 不包含：重定向到认证中心登录页面
     * 
     * @param servletRequest
     * @param servletResponse
     * @param policy
     * @return
     * @throws IOException
     * @throws ServletException
     */
    protected InterceptResult intercept(ServletRequest servletRequest,ServletResponse servletResponse,Configuration configuration) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        // 获取用户访问请求地址，此地址以“/”开头且不包含任何参数
        String requestUri = request.getRequestURI();
        // 获取应用编码
        String appCode = configuration.getSsoConfig().getAppcode();
        // 获取用户请求地址的域名
        String domain = RequestUtil.getDomainByRequest(request);
        // 判断是否是受限资源
        AccessPolicy policy = configuration.getSsoAccess().matchAccessPolicy(requestUri);
        // 定义验证结果
        InterceptResult interceptResult = null;
        if (AccessPolicy.LIMIT.equals(policy)) {
            // 验证是否是受限资源
        	boolean isWhiteList = SsoRemoteService.checkWhiteList(domain);
            if (isWhiteList) {
                interceptResult = this.limitIntercept(request, response, configuration);
            } else {
                interceptResult = new InterceptResult(false, isWhiteList, servletRequest);
            }
        } else if (AccessPolicy.PASS.equals(policy)) {
            interceptResult = this.passIntercept(request, response, configuration);
        } else if (AccessPolicy.UNDEFINED.equals(policy)) {
            interceptResult = new InterceptResult(true, false, new SsoHttpServletRequestWrapper(request, null));
        }
        return interceptResult;
    }

    /**
     * 
     * 功能描述: 校验受限资源<br>
     *
     * @param request
     * @param response
     * @param configuration
     * @return
     * @throws IOException
     * @throws ServletException
     */
    public InterceptResult limitIntercept(HttpServletRequest request,HttpServletResponse response,Configuration configuration) throws IOException, ServletException {
        InterceptResult interceptResult = null;
        SessionDTO sessionDTO = null;
        String domain = configuration.getSsoConfig().getCookieDomain();
        try {
            /**
             * 1.判断当前请求是Ajax请求 2.判断当前请求是否jsonP请求
             */
            String requestType = request.getHeader("X-Requested-With");
            // 获取url中的yncsid
            String yncsid = request.getParameter(ConstantUtil.YNCSID);
            if (null != yncsid && !"".equals(yncsid)) {
                sessionDTO = this.checkYncsid(yncsid, false, response, configuration, sessionDTO); //去缓存或中台检验改sid是否存在session（即是否登录过）
            } else {
                yncsid = CookieUtil.getCookieValue(request, ConstantUtil.YNCSID);
                if (null != yncsid && !"".equals(yncsid)) {
                    sessionDTO = this.checkYncsid(yncsid, true, response, configuration, sessionDTO);
                }
            }
            if (null != sessionDTO) {
                // 如果是https请求，则需要校验yncst
                if (request.isSecure()) {
                    String yncst = request.getParameter(ConstantUtil.YNCST);
                    if (!StringUtil.isEmpty(yncst) && yncst.equals(sessionDTO.getYncst())) {
                        interceptResult = new InterceptResult(true, true, new SsoHttpServletRequestWrapper(request, sessionDTO));
                    } else {
                        // 从cookie中获取yncst
                        yncst = CookieUtil.getCookieValue(request, ConstantUtil.YNCST);
                        if (StringUtil.isEmpty(yncst) || !yncst.equals(sessionDTO.getYncst())) {
                            CookieUtil.addCookie(response, ConstantUtil.YNCSID, "", domain, "/", false, 0);
                            CookieUtil.addCookie(response, ConstantUtil.YNCUID, "", domain, "/", false, 0);
                            CookieUtil.addCookie(response, ConstantUtil.YNCST, "", domain, "/", true, 0);
                            // 重定向到认证中心的登录页面
                            redirectToSsoLogin(request, response, configuration);
                            interceptResult = new InterceptResult(false, true, request);
                        }else{
                        	 interceptResult = new InterceptResult(true, true, new SsoHttpServletRequestWrapper(request, sessionDTO));
                        }
                    }
                } else {
                    interceptResult = new InterceptResult(true, true, new SsoHttpServletRequestWrapper(request, sessionDTO));
                }
            } else {//未登录则跳转
                // 判断请求类型
                if (null != requestType && "XMLHttpRequest".equals(requestType)) {
                    // ajax请求
                    redirectToSsoAjaxLogin(request, response, configuration, false);
                } else {
                    // 重定向到认证中心的登录页面
                    redirectToSsoLogin(request, response, configuration);
                }
                interceptResult = new InterceptResult(false, true, request);
            }
        } catch (Exception ex) {
            return new InterceptResult(true, true, request);
        }
        return interceptResult;
    }

    /**
     * 
     * 功能描述: 校验pass模式的资源<br>
     *
     * @param request
     * @param response
     * @param configuration
     * @return
     * @throws IOException
     * @throws ServletException
     */
    public InterceptResult passIntercept(HttpServletRequest request,HttpServletResponse response,Configuration configuration) throws IOException, ServletException {
        try {
            SessionDTO sessionDTO = null;
            // 获取url中的yncsid
            String yncsid = request.getParameter(ConstantUtil.YNCSID);
            if (StringUtil.isEmpty(yncsid)) {
                yncsid = CookieUtil.getCookieValue(request, ConstantUtil.YNCSID);
            }
            if (null != yncsid && !"".equals(yncsid)) {
                sessionDTO = this.initSessionDTO(configuration, yncsid);
            }
            return new InterceptResult(true, true, new SsoHttpServletRequestWrapper(request, sessionDTO));
        } catch (Exception ex) {
            return new InterceptResult(true, true, null);
        }
    }

    /**
     * 
     * 功能描述: 请求重定向到认证中心，进行登陆验证<br>
     *
     * @param request
     * @param response
     * @param configuration
     * @throws IOException
     * @throws ServletException
     */
    protected void redirectToSsoLogin(HttpServletRequest request,HttpServletResponse response,Configuration configuration) throws IOException, ServletException {
        String currentUrl = "";
        // 构造受限资源地址
        if (request.getMethod().equalsIgnoreCase("GET")) {
            currentUrl = RequestUtil.buildRequestUrl(request);
            currentUrl = RequestUtil.appendQueryStr(request, currentUrl);
            currentUrl = RequestUtil.removeParameter(currentUrl, ConstantUtil.YNCSID);
            currentUrl = RequestUtil.removeParameter(currentUrl, ConstantUtil.YNCST);
            currentUrl = RequestUtil.removeParameter(currentUrl, ConstantUtil.YNCUID);
        }
        // 构造票据认证服务地址
        String serviceUrl = configuration.getSsoConfig().getUsercenterServerUrl() + "/login.htm";
        serviceUrl = RequestUtil.addParam(serviceUrl, "service", currentUrl);
        serviceUrl = RequestUtil.addParam(serviceUrl, ConstantUtil.APPCODE, configuration.getSsoConfig().getAppcode());
        // 重定向到passport认证中心
        // response.sendRedirect(configuration.getPassportServerUrl() + "/login?service="+serviceUrl);
        response.sendRedirect(serviceUrl);
    }

    /**
     * 
     * 功能描述: 请求重定向到认证中心，进行登陆验证<br>
     *
     * @param request
     * @param response
     * @param configuration
     * @throws IOException
     * @throws ServletException
     */
    protected void redirectToSsoAjaxLogin(HttpServletRequest request,HttpServletResponse response,Configuration configuration,boolean isJsonp) throws IOException, ServletException {
        String currentUrl = "";
        // 构造受限资源地址
        if (request.getMethod().equalsIgnoreCase("GET")) {
            currentUrl = RequestUtil.buildRequestUrl(request);
            currentUrl = RequestUtil.appendQueryStr(request, currentUrl);

            currentUrl = RequestUtil.removeParameter(currentUrl, ConstantUtil.YNCSID);
            currentUrl = RequestUtil.removeParameter(currentUrl, ConstantUtil.YNCST);
            currentUrl = RequestUtil.removeParameter(currentUrl, ConstantUtil.YNCUID);
        }
        // 构造票据认证服务地址
        String serviceUrl = RequestUtil.addParam(configuration.getSsoConfig().getTicketAjaxServerUrl(), ConstantUtil.TARGET_URL_PARAM, URLEncoder.encode(currentUrl, "UTF-8"));
        serviceUrl = RequestUtil.addParam(serviceUrl, ConstantUtil.APPCODE, configuration.getSsoConfig().getAppcode());
        Map<String, Object> mapJson = new HashMap<String, Object>();
        ObjectMapper objectMapper = new ObjectMapper();
        String jsonfromMap = "";
        mapJson.put("isNotLogin", true);
        mapJson.put("outDomain", false);
        mapJson.put("serviceUrl", configuration.getSsoConfig().getUsercenterServerUrl() + "/loginAjax?service=" + serviceUrl);
        // passport登录校验地址
        String checkServiceUrl = configuration.getSsoConfig().getUsercenterServerUrl() + "/checkLoginAjax?";
        checkServiceUrl = RequestUtil.addParam(checkServiceUrl, ConstantUtil.TARGET_URL_PARAM, URLEncoder.encode(currentUrl, "UTF-8"));
        checkServiceUrl = RequestUtil.addParam(checkServiceUrl, ConstantUtil.APPCODE, configuration.getSsoConfig().getAppcode());
        mapJson.put("checkServiceUrl", checkServiceUrl);
        jsonfromMap = objectMapper.writeValueAsString(mapJson);
        if (isJsonp) {
            jsonfromMap = request.getParameter("callback") + "([" + jsonfromMap + "])";
        }
        response.setCharacterEncoding("UTF-8");
        response.setContentType("application/json; charset=utf-8");
        response.getWriter().print(jsonfromMap);
    }

//    /**
//     * 
//     * 功能描述: 此处需要根据访问的域名和appCode验证当前客户端是否在白名单中<br>
//     *
//     * @param appCode
//     * @param domain
//     * @param configuration
//     * @return
//     */
//    private boolean checkWhiteList(String appCode,String domain,Configuration configuration) {
//        boolean flagAppCode = false;
//        boolean flagDomain = false;
//        // 从缓存中读取应用列表,如果缓存为空，则从sso获取白名单信息
//        WhiteListDTO whiteListDTO = this.initWhiteListDTO(configuration, "WhiteListDTO");
//        if(null == whiteListDTO){
//        	return false;
//        }
//        List<String> appCodeList = whiteListDTO.getAcList();
//        List<String> domainList = whiteListDTO.getDomainList();
//        if (null != domainList && null != appCodeList) {
//            flagAppCode = appCodeList.contains(appCode);
//            flagDomain = domainList.contains(domain);
//        }
//        return (flagAppCode && flagDomain);
//    }

//    /**
//     * 
//     * 功能描述: 设置本地白名单缓存<br>
//     *
//     * @param configuration
//     * @param key
//     * @return
//     */
//    private WhiteListDTO initWhiteListDTO(Configuration configuration,String key) {
//        WhiteListDTO whiteListDTO = null;
//        String ehcacheKey = "ehcache" + key;
//        // 先从ehcache中获取对应的数据
//        if(DebugUtil.useEhCache){//TODO 使用缓存
//            Element element = configuration.getEhcacheClient().getCache().get(ehcacheKey);
//            if (null == element) {
//                synchronized (ehcacheKey.intern()) {
//                    element = configuration.getEhcacheClient().getCache().get(ehcacheKey);
//                    if (null == element) {
//                        try {
//                            whiteListDTO = configuration.getHessianClientFactory().getSsoService().getAllWhiteList();
//                        } catch (Exception e1) {
//                            e1.printStackTrace();
//                        }
//                        // 设置缓存
//                        try {
//                            element = new Element(ehcacheKey, whiteListDTO);
//                            configuration.getEhcacheClient().getCache().put(element);
//                        } catch (Exception e) {
//
//                        }
//                    }
//                }
//            } else {
//                whiteListDTO = (WhiteListDTO) element.getObjectValue();
//            }
//            if(null == whiteListDTO){
//                whiteListDTO = configuration.getHessianClientFactory().getSsoService().getAllWhiteList(); 
//            }
//        }else{
//            whiteListDTO = configuration.getHessianClientFactory().getSsoService().getAllWhiteList();
//        }
//        return whiteListDTO;
//    }

    /**
     * 
     * 功能描述: 校验yncsid<br>
     *
     * @param yncsid
     * @param isCookie 是否是cookie中的yncsid true：是 false：否
     * @return
     */
    private SessionDTO checkYncsid(String yncsid,boolean isCookie,HttpServletResponse response,Configuration configuration,SessionDTO sessionDTO) {
        String domain = configuration.getSsoConfig().getCookieDomain();
        if (null != yncsid && !"".equals(yncsid)) {
            sessionDTO = this.initSessionDTO(configuration, yncsid);
            if (null != sessionDTO) {
                if (!isCookie) {
                    CookieUtil.addCookie(response, ConstantUtil.YNCSID, sessionDTO.getYncsid(), domain, "/", false, null);
                    CookieUtil.addCookie(response, ConstantUtil.YNCUID, sessionDTO.getYnctgt().getYncuid(), domain, "/", false, 1800);
                    CookieUtil.addCookie(response, ConstantUtil.YNCST, sessionDTO.getYncst(), domain, "/", true, null);
                }
            } else {//情况cookie中的值
                CookieUtil.addCookie(response, ConstantUtil.YNCSID, "", domain, "/", false, 0);
                CookieUtil.addCookie(response, ConstantUtil.YNCUID, "", domain, "/", false, 0);
                CookieUtil.addCookie(response, ConstantUtil.YNCST, "", domain, "/", true, 0);
            }
        }
        return sessionDTO;
    }

    /**
     * 
     * 功能描述: 设施SessionDTO本地缓存<br>
     *
     * @param configuration
     * @param key
     * @return
     */
    private SessionDTO initSessionDTO(Configuration configuration, String yncsid) {
        SessionDTO sessionDTO = null;
        // 先从ehcache中获取对应的数据
        Element element = configuration.getEhcacheClient().getCache().get(yncsid);
        if (null == element) {
            synchronized (yncsid.intern()) {
                element = configuration.getEhcacheClient().getCache().get(yncsid);
                if (null == element) {
//                    String sessionKey = CacheUtil.YNCSID_KEY + yncsid;
//                    sessionDTO = JacksonMapperUtil.jsonToBean(configuration.getRedisClient().get(sessionKey), SessionDTO.class);
//                    configuration.getRedisClient().expire(sessionKey, 300);
                	SessionDTO tmp = SsoRemoteService.getSession(yncsid);
                    
                    if(tmp!=null && tmp.success){
                    	sessionDTO = tmp;
                        try {
                            // 设置缓存
                            element = new Element(yncsid, sessionDTO);
                            configuration.getEhcacheClient().getCache().put(element);
                        } catch (Exception e) {

                        }
                    }
                }
            }
        } else {
            sessionDTO = (SessionDTO) element.getObjectValue();
        }
        return sessionDTO;
    }
}
